#!/usr/bin/python3
# -*- coding: utf-8 -*-
""" module for external apps calling"""

import os
import subprocess
import shutil
import time
import struct

import sgconf
import sgutils


def get_image_size(fname):
    """ return the image sizes: wrapper for single type
         extensions, to keep separated """

    myext = os.path.splitext(fname)[1]

    match myext:
        case ".png":
            return get_png_size(fname)
        case ".gif":
            return get_gif_size(fname)
        case ".jpg" | ".jpeg":
            return get_jpg_size(fname)
        case ".webp":
            return get_webp_size(fname)
        case _:
            return [1, 1]


def get_gif_size(file_path):
    """ return the image sizes of a gif file """

    with open(file_path, "rb") as fhandle:    
        head = fhandle.read(24)
        if len(head) != 24:
            return [1, 1]
    
        width, height = struct.unpack('<HH', head[6:10])
        
    return [width, height]


def get_png_size(file_path):
    """ return the png image sizes """

    with open(file_path, "rb") as fhandle:
        head = fhandle.read(24)
        if len(head) != 24:
            return [1, 1]
        
        check = struct.unpack(">i", head[4:8])[0]
        if check != 0x0d0a1a0a:
            return [1, 1]
        width, height = struct.unpack('>ii', head[16:24])

    return [width, height]

    
def get_jpg_size(file_path):
    """ return the image sizes for jpg files """

    with open(file_path, "rb") as fhandle:
        head = fhandle.read(24)
        if len(head) != 24:
            return [1, 1]
        try:
            fhandle.seek(0)  # Read 0xff next
            size = 2
            ftype = 0
            while not 0xc0 <= ftype <= 0xcf:
                fhandle.seek(size, 1)
                byte = fhandle.read(1)
                while ord(byte) == 0xff:
                    byte = fhandle.read(1)
                ftype = ord(byte)
                size = struct.unpack('>H', fhandle.read(2))[0] - 2
            # We are at a SOFn block
            fhandle.seek(1, 1)  # Skip `precision' byte.
            height, width = struct.unpack('>HH', fhandle.read(4))
        except Exception:    
            return [1, 1]

    return [width, height]
    


def get_webp_size(file_path):
    """ read the size in pixel of a webp file """
    with open(file_path, 'rb') as f:
        riff_header = f.read(12)  # Legge i primi 12 byte (RIFF header)
        
        if riff_header[:4] != b'RIFF' or riff_header[8:12] != b'WEBP':
            return [1, 1]

        webp_header = f.read(4)

        if webp_header == b'VP8 ': # WebP Lossy (VP8)
            f.read(10)
            width, height = f.read(2), f.read(2)
            width = int.from_bytes(width, 'little') & 0x3FFF  # 14 bit
            height = int.from_bytes(height, 'little') & 0x3FFF # 14 bit
        elif webp_header == b'VP8L': # WebP Lossless (VP8L)
            f.read(1)
            b1, b2, b3, b4 = f.read(1), f.read(1), f.read(1), f.read(1)
            width = 1 + ((b1[0] | ((b2[0] & 0x3F) << 8)) & 0x3FFF)
            height = 1 + ((((b2[0] >> 6) | (b3[0] << 2)) | ((b4[0] & 0x0F) << 10)) & 0x3FFF)
        elif webp_header == b'VP8X': # WebP Extended (VP8X)
            f.read(8)
            width, height = f.read(3), f.read(3)
            width = (int.from_bytes(width, 'little') & 0xFFFFFF) + 1
            height = (int.from_bytes(height, 'little') & 0xFFFFFF) + 1
        else:
            return [1, 1]

    return [width, height]
    
    
def extbackup(fn):
    # noinspection PyPep8
    """backup an image should copy a backup of the modified images in a directory, and adding a number at the end if it is
        already present a file with that name
        :param fn: filename
        :return: none
        """

    if sgconf.cfgget("dirbackup") != "":
        today = time.strftime("%Y-%m-%d")
        db = os.path.join(sgconf.cfgget("dirbackup"), today)
        if not os.path.exists(db):
            os.makedirs(db)

        if os.path.exists(db):
            newfile = os.path.join(db, os.path.basename(fn))
            if not os.path.exists(newfile):
                shutil.copy(fn, newfile)
            else:
                for numero in range(1, 99):
                    newfile = os.path.join(db, os.path.basename(fn) + "." + str(numero))
                    if not os.path.exists(newfile):
                        shutil.copy(fn, newfile)
                        return


def extgetcmd(comando):
    """ run a program and get the results
    :param comando: the command line that will be launched
    :return: output of the command line
    """
    if os.name == "nt":
        comando = comando.replace("'", "\"")

    arr = []
    p = subprocess.Popen(comando, shell=True, stdout=subprocess.PIPE, stderr=subprocess.STDOUT)
    for line in p.stdout.readlines():
        try:
            arr.append(line.strip().decode('ascii'))
        except Exception:
            try:
                arr.append(line.strip().decode('utf-8'))
            except Exception:
                arr.append(line.strip().decode('ascii', 'ignore'))
        p.wait()

    return "\n".join(arr)


def extgraphicinfo(immagine, cosa):
    """ extracts the information for image, also using a pure Python
        mode reading the file characters
    :param immagine: name of image file
    :param cosa: what to extract linksize=800x600 arrdim=[800, 600]
    :return:
    """

    if cosa == "linksize":
        arr = get_image_size(immagine)
        v = f"width='{arr[0]}' height='{arr[1]}'"
        return v
    elif cosa == "exif":
        cmdl = sgconf.cfgget("appidentify") + " -format %[exif:*] '" + immagine + "'"
        v = extgetcmd(cmdl)
        return v
    elif cosa == "arrdim":
        arr = get_image_size(immagine)
        return arr


def extgraphicresize(immagine, larghezza, thumb):
    """imagemagick command that resizes images

    :param immagine: image file
    :param larghezza: width
    :param thumb: is a thumbnail?
    :return: none
    """

    compressione = sgconf.cfgget("imagescompression")
    densita = sgconf.cfgget("imagesresolution")
    if thumb == "":
        extbackup(immagine)
        cmdline = sgconf.cfgget("appmogrify") + ' ' + sgconf.cfgget("imagestriptags") + ' -density ' + str(densita) + ' -quality ' + str(compressione) + ' -scale ' + str(larghezza) + ' "' + immagine + '"'
        extruncmd(cmdline, True)
    else:
        cmdline = sgconf.cfgget("appconvert") + ' ' + sgconf.cfgget("imagestriptags") + ' -quality ' + str(compressione) + ' -scale ' + str(larghezza) + '  "' + immagine + '" "' + thumb + '"'
        extruncmd(cmdline, True)
        # this will resize thumbnails if width < height
        # arr = extgraphicinfo(thumb, "arrdim")
        arr = get_image_size(immagine)
        w = int(arr[0])
        h = int(arr[1])
        if w < h:
            cmdline = sgconf.cfgget("appmogrify") + ' ' + sgconf.cfgget("imagestriptags") + ' -quality ' + str(compressione) + ' -crop ' + str(larghezza) + 'x' + str(larghezza) + '+0-' + str(round((h - w) / 2)) + ' "' + thumb + '"'
            extruncmd(cmdline, True)  #


def extruncmd(comando, showoutput):
    """
    :param comando:
    :param showoutput:
    :return:
    run a program without getting results """
    if os.name == "nt":
        comando = comando.replace("'", "\"")

    arr = []
    p = subprocess.Popen(comando, shell=True, stdout=subprocess.PIPE, stderr=subprocess.STDOUT)
    if showoutput:
        for line in p.stdout.readlines():
            arr.append(line.strip().decode('ascii'))
        sgutils.showmsg("  executing:\n" + comando, 0)
        p.wait()
